import logging
from flask import Flask, request, jsonify, render_template
import fitz  # PyMuPDF
import cv2
import numpy as np
from pyzbar.pyzbar import decode
import pandas as pd
import os
import tempfile

# Set up logging to a specific log file
log_file = '/home/artistsaafnotick/public_html/myapp/logs/flask_error.log'

logging.basicConfig(
    filename=log_file,
    level=logging.DEBUG,
    format='%(asctime)s - %(levelname)s - %(message)s'
)

app = Flask(__name__)

# ---------- QR PROCESSING FUNCTIONS ----------

def process_image(image):
    """Process image to detect and decode QR codes"""
    gray_image = cv2.cvtColor(image, cv2.COLOR_BGR2GRAY)
    blurred_image = cv2.GaussianBlur(gray_image, (5, 5), 0)
    _, thresh_image = cv2.threshold(blurred_image, 0, 255, cv2.THRESH_BINARY + cv2.THRESH_OTSU)

    qr_codes = decode(thresh_image)
    decoded_texts = [qr_code.data.decode("utf-8") for qr_code in qr_codes]

    return decoded_texts


def process_pdf(pdf_path, scale_factor=5):
    """Convert PDF pages to images and scan for QR codes"""
    doc = fitz.open(pdf_path)
    all_decoded_texts = []

    for page_num in range(doc.page_count):
        page = doc.load_page(page_num)
        matrix = fitz.Matrix(scale_factor, scale_factor)
        pix = page.get_pixmap(matrix=matrix)

        img = np.frombuffer(pix.samples, dtype=np.uint8)

        if pix.n < 4:
            img = img.reshape(pix.height, pix.width, 3)  # RGB
        else:
            img = img.reshape(pix.height, pix.width, 4)  # RGBA
            img = cv2.cvtColor(img, cv2.COLOR_RGBA2BGR)

        decoded_texts = process_image(img)
        all_decoded_texts.extend(decoded_texts)

    return all_decoded_texts

# ---------- FLASK ROUTES ----------

@app.route("/")
def index():
    return render_template("index.html")


@app.route("/upload", methods=["POST"])
def upload_pdf():
    if "file" not in request.files:
        logging.error("No file uploaded")
        return jsonify({"error": "No file uploaded"}), 400

    file = request.files["file"]

    if not file.filename.lower().endswith(".pdf"):
        logging.error(f"Invalid file type: {file.filename}")
        return jsonify({"error": "Only PDF files allowed"}), 400

    try:
        # Save file temporarily
        with tempfile.NamedTemporaryFile(delete=False, suffix=".pdf") as tmp_file:
            file.save(tmp_file.name)
            pdf_path = tmp_file.name

        # Process PDF
        decoded_texts = process_pdf(pdf_path)

        # Save CSV inside static folder to allow download
        if decoded_texts:
            csv_filename = "decoded_qr_codes.csv"
            csv_path = os.path.join(app.static_folder, csv_filename)
            df = pd.DataFrame(decoded_texts, columns=["QR Code Text"])
            df.to_csv(csv_path, index=False, encoding="utf-8-sig")
            result = {
                "decoded_qr_codes": decoded_texts,
                "csv_saved": True,
                "csv_url": f"/myapp/static/{csv_filename}"
            }
        else:
            result = {"decoded_qr_codes": [], "csv_saved": False}

    except Exception as e:
        # Log the error
        logging.error(f"Error processing PDF: {e}")
        return jsonify({"error": f"Failed to process PDF: {str(e)}"}), 500

    finally:
        try:
            os.remove(pdf_path)
        except Exception as e:
            logging.error(f"Error deleting temp PDF file: {e}")
            pass

    return jsonify(result)

# ---------- WSGI ENTRY POINT ----------
if __name__ == "__main__":
    app.run(debug=True)
